/*
* ms_complex_clk.c- Sigmastar
*
* Copyright (C) 2018 Sigmastar Technology Corp.
*
* Author: karl.xiao <karl.xiao@sigmastar.com.tw>
*
* This software is licensed under the terms of the GNU General Public
* License version 2, as published by the Free Software Foundation, and
* may be copied, distributed, and modified under those terms.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
*/
#include <linux/kernel.h>
#include <linux/vmalloc.h>
#include <linux/of.h>
#include "registers.h"
#include "voltage_ctrl.h"

#if defined (CONFIG_MS_CPU_FREQ) && defined (CONFIG_MS_GPIO)
extern u8 enable_scaling_voltage;
#include "gpio.h"
#include "../../gpio/mdrv_gpio.h"
#else
u8 enable_scaling_voltage=0;
#define MDrv_GPIO_Set_High(x) {}
#define MDrv_GPIO_Set_Low(x) {}
#define MDrv_GPIO_Pad_Set(x) {}
#endif

#define VOLCTRL_DEBUG  0

#if VOLCTRL_DEBUG
#define VOLCTRL_DBG(fmt, arg...) printk(KERN_INFO fmt, ##arg)
#else
#define VOLCTRL_DBG(fmt, arg...)
#endif
#define VOLCTRL_ERR(fmt, arg...) printk(KERN_ERR fmt, ##arg)

unsigned int gpio_vid_width = 0;
unsigned int *gpio_vid_pins = NULL;
unsigned int *gpio_vid_voltages = NULL;
unsigned int gpio_vid_default = 0;

int g_sCurrentVoltageCore = 0;

int s_VoltageFactorWishBox[VOLTAGE_DEMANDER_MAX] = {0};

void set_core_voltage(VOLTAGE_DEMANDER_E demander, int mV)
{
    int voltage_lv_cnt = 0;
    int gpio_lv_cnt = 0;
    int i = 0;
    int vcore_max = 0;
    int vcore_total_lv = 1 << gpio_vid_width;

    // Check core voltage setting(mV) valid or not.
    for (voltage_lv_cnt = 0; voltage_lv_cnt < vcore_total_lv; voltage_lv_cnt++)
    {
        if (mV == gpio_vid_voltages[voltage_lv_cnt])
        {
            break;
        }
    }

    if (voltage_lv_cnt == vcore_total_lv)  // If core voltage setting(mV) is invalid, Try to find ceiling setting.
    {
        for (voltage_lv_cnt = 0; voltage_lv_cnt < vcore_total_lv; voltage_lv_cnt++)
        {
            if (mV < gpio_vid_voltages[voltage_lv_cnt])
            {
                VOLCTRL_ERR("[%s] Not support %dmV, update demander%d's request to %dmV\n",
                            __func__, mV, demander, gpio_vid_voltages[voltage_lv_cnt]);
                mV = gpio_vid_voltages[voltage_lv_cnt];
                break;
            }
        }

        if (voltage_lv_cnt == vcore_total_lv)  // If no ceiling setting, use highest setting.
        {
            voltage_lv_cnt--;
            VOLCTRL_ERR("[%s] Not support %dmV, update demander%d's request to %dmV",
                        __func__, mV, demander, gpio_vid_voltages[voltage_lv_cnt]);
            mV = gpio_vid_voltages[voltage_lv_cnt];
        }
    }

    if (demander < VOLTAGE_DEMANDER_MAX)
    {
        s_VoltageFactorWishBox[demander] = mV;
        for (i=0; i<VOLTAGE_DEMANDER_MAX; i++)
        {
            vcore_max = max(vcore_max, s_VoltageFactorWishBox[i]);
        }
    }
    else if (demander == VOLTAGE_DEMANDER_INIT)
    {
        vcore_max = mV;
    }
    else
    {
        return;
    }

    if(enable_scaling_voltage)
    {
        if(gpio_vid_width > 0)
        {
            for (voltage_lv_cnt = 0; voltage_lv_cnt < vcore_total_lv; voltage_lv_cnt++)
            {
                if (vcore_max == gpio_vid_voltages[voltage_lv_cnt])
                {
                    for (gpio_lv_cnt=0; gpio_lv_cnt<gpio_vid_width; gpio_lv_cnt++)
                    {
                        if ((voltage_lv_cnt >> gpio_lv_cnt) & 0x1)
                        {
                            MDrv_GPIO_Set_High(gpio_vid_pins[gpio_lv_cnt]);
                        }
                        else
                        {
                            MDrv_GPIO_Set_Low(gpio_vid_pins[gpio_lv_cnt]);
                        }
                    }

                    g_sCurrentVoltageCore = gpio_vid_voltages[voltage_lv_cnt];
                    break;
                }
            }
        }
    }
    VOLCTRL_DBG("CurrentVoltageCore = %d\n", g_sCurrentVoltageCore);
}
EXPORT_SYMBOL(set_core_voltage);

int get_core_voltage(void)
{
    return g_sCurrentVoltageCore;
}
EXPORT_SYMBOL(get_core_voltage);

void init_core_voltage(void)
{
    struct device_node *np = NULL;
    int gpio_lv_cnt = 0;

    if((np=of_find_node_by_name(NULL, "core_voltage")))
    {
        if (0 != of_property_read_u32(np, "vid_width", &gpio_vid_width) || gpio_vid_width < 1)
            goto core_voltage_init_err;

        if (0 != of_property_read_u32(np, "vid_default", &gpio_vid_default))
            goto core_voltage_init_err;

        gpio_vid_pins = vzalloc(sizeof(unsigned int) * gpio_vid_width);
        if (!gpio_vid_pins)
            goto core_voltage_init_err;

        gpio_vid_voltages = vzalloc(sizeof(unsigned int) * (1 << gpio_vid_width));
        if (!gpio_vid_voltages)
            goto core_voltage_init_err;

        if (gpio_vid_width != of_property_read_variable_u32_array(np, "vid_gpios", gpio_vid_pins, 0, gpio_vid_width))
            goto core_voltage_init_err;
        if ((1 << gpio_vid_width) != of_property_read_variable_u32_array(np, "vid_voltages", gpio_vid_voltages, 0, (1 << gpio_vid_width)))
                goto core_voltage_init_err;

        for (gpio_lv_cnt=0; gpio_lv_cnt<gpio_vid_width; gpio_lv_cnt++)
        {
            MDrv_GPIO_Pad_Set(gpio_vid_pins[gpio_lv_cnt]);
            if ((gpio_vid_default >> gpio_lv_cnt) & 0x1)
            {
                MDrv_GPIO_Set_High(gpio_vid_pins[gpio_lv_cnt]);
            }
            else
            {
                MDrv_GPIO_Set_Low(gpio_vid_pins[gpio_lv_cnt]);
            }
        }

        g_sCurrentVoltageCore = gpio_vid_voltages[gpio_vid_default];
    }
    else
    {
        VOLCTRL_ERR("[%s] can't get core_voltage node for voltage ctrl\n", __func__);
    }

    return;

core_voltage_init_err:
    gpio_vid_width = 0;
    gpio_vid_default = 0;
    if (gpio_vid_pins)
    {
        vfree(gpio_vid_pins);
        gpio_vid_pins = NULL;
    }
    if (gpio_vid_voltages)
    {
        vfree(gpio_vid_voltages);
        gpio_vid_voltages = NULL;
    }
    return;
}
EXPORT_SYMBOL(init_core_voltage);

void deinit_core_voltage(void)
{
    gpio_vid_width = 0;
    gpio_vid_default = 0;

    if (gpio_vid_pins)
    {
        vfree(gpio_vid_pins);
        gpio_vid_pins = NULL;
    }
    if (gpio_vid_voltages)
    {
        vfree(gpio_vid_voltages);
        gpio_vid_voltages = NULL;
    }

    return;
}
EXPORT_SYMBOL(deinit_core_voltage);

void reset_core_voltage(void)
{
    int gpio_lv_cnt = 0;

    if (gpio_vid_width > 0)
    {
        for (gpio_lv_cnt=0; gpio_lv_cnt<gpio_vid_width; gpio_lv_cnt++)
        {
            if ((gpio_vid_default >> gpio_lv_cnt) & 0x1)
            {
                MDrv_GPIO_Set_High(gpio_vid_pins[gpio_lv_cnt]);
            }
            else
            {
                MDrv_GPIO_Set_Low(gpio_vid_pins[gpio_lv_cnt]);
            }
        }

        g_sCurrentVoltageCore = gpio_vid_voltages[gpio_vid_default];
    }
}
EXPORT_SYMBOL(reset_core_voltage);

int core_voltage_available(unsigned int **voltages, unsigned int *num)
{
    if (voltages && num)
    {
        *num = 1 << gpio_vid_width;
        *voltages = gpio_vid_voltages;
        return 0;
    }
    else
    {
        return -1;
    }
}
EXPORT_SYMBOL(core_voltage_available);

int core_voltage_pin(unsigned int **pins, unsigned int *num)
{
    if (pins && num)
    {
        *num = gpio_vid_width;
        *pins = gpio_vid_pins;
        return 0;
    }
    else
    {
        return -1;
    }
}
EXPORT_SYMBOL(core_voltage_pin);
